<?php

declare(strict_types=1);

namespace App\Models;

use PDO;

class Transaction
{
    public function __construct(
        private PDO $pdo
    ) {}

    public function findByGameRound(string $gameRound): ?array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM transactions WHERE game_round = ?');
        $stmt->execute([$gameRound]);
        return $stmt->fetch() ?: null;
    }

    public function updateWin(string $gameRound, float $winAmount, float $newAfterBalance): bool
    {
        $stmt = $this->pdo->prepare('UPDATE transactions SET win_amount = ?, after_balance = ? WHERE game_round = ?');
        $stmt->execute([$winAmount, $newAfterBalance, $gameRound]);
        return $stmt->rowCount() > 0;
    }

    public function updateBetAndWin(string $gameRound, float $betAmount, float $winAmount, float $newAfterBalance): bool
    {
        $stmt = $this->pdo->prepare('UPDATE transactions SET bet_amount = ?, win_amount = ?, after_balance = ? WHERE game_round = ?');
        $stmt->execute([$betAmount, $winAmount, $newAfterBalance, $gameRound]);
        return $stmt->rowCount() > 0;
    }

    public function create(array $data): int
    {
        $stmt = $this->pdo->prepare('
            INSERT INTO transactions (client_id, user_id, game_uid, game_round, bet_amount, win_amount, before_balance, after_balance, commission_amount)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([
            $data['client_id'],
            $data['user_id'],
            $data['game_uid'],
            $data['game_round'],
            $data['bet_amount'],
            $data['win_amount'],
            $data['before_balance'],
            $data['after_balance'],
            $data['commission_amount'] ?? 0,
        ]);
        return (int) $this->pdo->lastInsertId();
    }

    public function getList(?int $clientId = null, ?string $dateFrom = null, ?string $dateTo = null, int $limit = 100, int $offset = 0): array
    {
        $sql = 'SELECT t.*, c.name as client_name FROM transactions t JOIN clients c ON c.id = t.client_id WHERE 1=1';
        $params = [];
        if ($clientId) {
            $sql .= ' AND t.client_id = ?';
            $params[] = $clientId;
        }
        if ($dateFrom) {
            $sql .= ' AND t.created_at >= ?';
            $params[] = $dateFrom . ' 00:00:00';
        }
        if ($dateTo) {
            $sql .= ' AND t.created_at <= ?';
            $params[] = $dateTo . ' 23:59:59';
        }
        $sql .= ' ORDER BY t.created_at DESC LIMIT ? OFFSET ?';
        $params[] = $limit;
        $params[] = $offset;
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll();
    }

    public function getStats(?int $clientId = null): array
    {
        $sql = 'SELECT COALESCE(SUM(bet_amount), 0) as total_bets, COALESCE(SUM(win_amount), 0) as total_wins, COALESCE(SUM(commission_amount), 0) as total_commission FROM transactions WHERE 1=1';
        $params = [];
        if ($clientId) {
            $sql .= ' AND client_id = ?';
            $params[] = $clientId;
        }
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $row = $stmt->fetch();
        $bets = (float) $row['total_bets'];
        $wins = (float) $row['total_wins'];
        $commission = (float) $row['total_commission'];
        return [
            'total_bets' => $bets,
            'total_wins' => $wins,
            'total_commission' => $commission,
            'total_profit' => $bets - $wins - $commission,
        ];
    }
}
