<?php

declare(strict_types=1);

namespace App\Models;

use PDO;

class GameSession
{
    public function __construct(
        private PDO $pdo
    ) {}

    public function create(int $clientId, string $userId, string $gameUid, ?string $sessionId = null, ?string $callbackUrl = null, ?string $returnUrl = null): void
    {
        $stmt = $this->pdo->prepare('
            INSERT INTO game_sessions (client_id, user_id, game_uid, session_id, callback_url, return_url)
            VALUES (?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([$clientId, $userId, $gameUid, $sessionId, $callbackUrl, $returnUrl]);
    }

    /** Create or update session - game_uid is game code, can repeat for same user relaunch */
    public function createOrUpdate(int $clientId, string $userId, string $gameUid, ?string $sessionId, ?string $callbackUrl, ?string $returnUrl): void
    {
        $stmt = $this->pdo->prepare('
            INSERT INTO game_sessions (client_id, user_id, game_uid, session_id, callback_url, return_url)
            VALUES (?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE session_id = VALUES(session_id), callback_url = VALUES(callback_url), return_url = VALUES(return_url)
        ');
        $stmt->execute([$clientId, $userId, $gameUid, $sessionId, $callbackUrl, $returnUrl]);
    }

    public function findByGameUid(string $gameUid): ?array
    {
        $stmt = $this->pdo->prepare('
            SELECT gs.*, c.callback_url as client_callback_url, c.return_url as client_return_url
            FROM game_sessions gs
            JOIN clients c ON c.id = gs.client_id
            WHERE gs.game_uid = ?
            ORDER BY gs.created_at DESC LIMIT 1
        ');
        $stmt->execute([$gameUid]);
        return $stmt->fetch() ?: null;
    }

    /** Find session by game_uid + user (member_account) - needed when game_uid repeats across users */
    public function findByGameUidAndUser(string $gameUid, string $userId): ?array
    {
        $stmt = $this->pdo->prepare('
            SELECT gs.*, c.callback_url as client_callback_url, c.return_url as client_return_url
            FROM game_sessions gs
            JOIN clients c ON c.id = gs.client_id
            WHERE gs.game_uid = ? AND gs.user_id = ?
            ORDER BY gs.created_at DESC LIMIT 1
        ');
        $stmt->execute([$gameUid, $userId]);
        return $stmt->fetch() ?: null;
    }

    /** Fallback: find most recent session for user (in case callback game_uid format differs) */
    public function findLatestByUser(string $userId, int $maxMinutes = 60): ?array
    {
        $stmt = $this->pdo->prepare('
            SELECT gs.*, c.callback_url as client_callback_url, c.return_url as client_return_url
            FROM game_sessions gs
            JOIN clients c ON c.id = gs.client_id
            WHERE gs.user_id = ? AND gs.created_at >= DATE_SUB(NOW(), INTERVAL ? MINUTE)
            ORDER BY gs.created_at DESC LIMIT 1
        ');
        $stmt->execute([$userId, $maxMinutes]);
        return $stmt->fetch() ?: null;
    }
}
