<?php

declare(strict_types=1);

namespace App\Models;

use PDO;

class Client
{
    public function __construct(
        private PDO $pdo
    ) {}

    public function findByApiKey(string $apiKey): ?array
    {
        $apiKey = trim($apiKey);
        if ($apiKey === '') return null;
        $stmt = $this->pdo->prepare('
            SELECT id, name, api_key, status, allowed_ips, allowed_domains,
                   callback_url, return_url, balance, commission_percent, created_at
            FROM clients WHERE BINARY api_key = ? AND LOWER(TRIM(status)) = ?
        ');
        $stmt->execute([$apiKey, 'active']);
        $row = $stmt->fetch();
        if (!$row) return null;
        return $this->hydrate($row);
    }

    public function findByApiKeyAnyStatus(string $apiKey): ?array
    {
        $apiKey = trim($apiKey);
        if ($apiKey === '') return null;
        $stmt = $this->pdo->prepare('SELECT id, api_key, status FROM clients WHERE BINARY api_key = ?');
        $stmt->execute([$apiKey]);
        return $stmt->fetch() ?: null;
    }

    public function getById(int $id): ?array
    {
        $stmt = $this->pdo->prepare('SELECT * FROM clients WHERE id = ?');
        $stmt->execute([$id]);
        $row = $stmt->fetch();
        return $row ? $this->hydrate($row) : null;
    }

    public function findAll(): array
    {
        $stmt = $this->pdo->query('SELECT * FROM clients ORDER BY created_at DESC');
        return array_map(fn($r) => $this->hydrate($r), $stmt->fetchAll());
    }

    public function create(array $data): int
    {
        $stmt = $this->pdo->prepare('
            INSERT INTO clients (name, api_key, status, allowed_ips, allowed_domains, callback_url, return_url, commission_percent)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ');
        $stmt->execute([
            $data['name'] ?? '',
            trim((string) ($data['api_key'] ?? '')),
            $data['status'] ?? 'active',
            is_array($data['allowed_ips'] ?? null) ? json_encode($data['allowed_ips']) : ($data['allowed_ips'] ?? null),
            is_array($data['allowed_domains'] ?? null) ? json_encode($data['allowed_domains']) : ($data['allowed_domains'] ?? null),
            $data['callback_url'] ?? null,
            $data['return_url'] ?? null,
            $data['commission_percent'] ?? 0,
        ]);
        return (int) $this->pdo->lastInsertId();
    }

    public function update(int $id, array $data): bool
    {
        $fields = [];
        $params = [];
        $jsonFields = ['allowed_ips', 'allowed_domains'];
        foreach (['name', 'api_key', 'status', 'allowed_ips', 'allowed_domains', 'callback_url', 'return_url', 'commission_percent'] as $f) {
            if (array_key_exists($f, $data)) {
                $fields[] = "$f = ?";
                $params[] = in_array($f, $jsonFields) && is_array($data[$f]) ? json_encode($data[$f]) : ($f === 'api_key' ? trim((string) ($data[$f] ?? '')) : $data[$f]);
            }
        }
        if (empty($fields)) return true;
        $params[] = $id;
        $stmt = $this->pdo->prepare('UPDATE clients SET ' . implode(', ', $fields) . ' WHERE id = ?');
        return $stmt->execute($params);
    }

    public function generateApiKey(): string
    {
        return bin2hex(random_bytes(32));
    }

    public function getStats(): array
    {
        $stmt = $this->pdo->query('
            SELECT COUNT(*) as total_clients,
                   (SELECT COALESCE(SUM(bet_amount), 0) FROM transactions) as total_bets,
                   (SELECT COALESCE(SUM(win_amount), 0) FROM transactions) as total_wins,
                   (SELECT COALESCE(SUM(bet_amount - win_amount - commission_amount), 0) FROM transactions) as total_profit
            FROM clients
        ');
        return $stmt->fetch();
    }

    private function hydrate(array $row): array
    {
        $row['allowed_ips'] = !empty($row['allowed_ips']) ? json_decode($row['allowed_ips'], true) : null;
        $row['allowed_domains'] = !empty($row['allowed_domains']) ? json_decode($row['allowed_domains'], true) : null;
        return $row;
    }
}
