<?php

declare(strict_types=1);

namespace App\Middleware;

use App\Services\HttpService;

class ForwardToMainServer
{
    public function __construct(
        private HttpService $http,
        private string $baseUrl,
        private string $apiKey,
        private string $proxyBaseUrl,
        private string $launchPath = '/api/launch-game',
    ) {}

    /**
     * Forward launch-game request to main server.
     * Uses client-server's own callback URL so main server sends callbacks here.
     *
     * @param array{user_id: string, balance: float, game_uid: string, return_url?: string, callback_url?: string, currency_code?: string, language?: string} $params
     * @return array{success: bool, body: string|null, code: int, decoded?: array, error?: string}
     */
    public function __invoke(array $params): array
    {
        if ($this->baseUrl === '' || $this->apiKey === '') {
            return [
                'success' => false,
                'body' => null,
                'code' => 500,
                'error' => 'MAIN_SERVER_URL and MAIN_SERVER_API_KEY must be set in .env',
                'decoded' => ['msg' => 'Server misconfigured. Contact admin.'],
            ];
        }

        $userId = trim((string) ($params['user_id'] ?? ''));
        $balance = (float) ($params['balance'] ?? 0);
        $gameUid = trim((string) ($params['game_uid'] ?? ''));
        $returnUrl = trim((string) ($params['return_url'] ?? $params['return'] ?? ''));
        $currencyCode = trim((string) ($params['currency_code'] ?? ''));
        $language = trim((string) ($params['language'] ?? ''));

        $ourCallbackUrl = rtrim($this->proxyBaseUrl, '/') . '/api/callback';

        $query = [
            'api_key' => $this->apiKey,
            'user_id' => $userId,
            'balance' => $balance,
            'game_uid' => $gameUid,
            'return'   => $returnUrl,
            'callback' => $ourCallbackUrl,
        ];
        if ($currencyCode !== '') {
            $query['currency_code'] = $currencyCode;
        }
        if ($language !== '') {
            $query['language'] = $language;
        }

        $baseUrl = $this->baseUrl;
        $parsed = parse_url($baseUrl);
        if (isset($parsed['path']) && $parsed['path'] !== '' && $parsed['path'] !== '/') {
            $baseUrl = ($parsed['scheme'] ?? 'https') . '://' . ($parsed['host'] ?? '') . (isset($parsed['port']) ? ':' . $parsed['port'] : '');
        }
        $path = rtrim($this->launchPath, '/') ?: '/api/launch-game';
        if (strpos($path, '?') !== false) {
            $path = strstr($path, '?', true) ?: $path;
        }
        $url = rtrim($baseUrl, '/') . $path . '?' . http_build_query($query);
        $result = $this->http->get($url);

        if (($result['code'] ?? 0) === 404 && isset($result['body']) && is_string($result['body']) && str_contains($result['body'], '404')) {
            $fallbackPath = '/api/launch-game.php';
            if ($path !== $fallbackPath) {
                $fallbackUrl = rtrim($baseUrl, '/') . $fallbackPath . '?' . http_build_query($query);
                $fallback = $this->http->get($fallbackUrl);
                if (($fallback['code'] ?? 0) !== 404 || !str_contains((string) ($fallback['body'] ?? ''), '404')) {
                    return $fallback;
                }
            }
        }

        return $result;
    }
}
