<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Middleware\CheckWhitelist;
use App\Middleware\ForwardToMainServer;
use App\Middleware\RateLimiter;
use App\Middleware\ValidateApiKey;
use App\Models\GameSession;
use App\Models\User;
use App\Services\Logger;

class LaunchGameController
{
    public function __construct(
        private ValidateApiKey $validateApiKey,
        private CheckWhitelist $checkWhitelist,
        private RateLimiter $rateLimiter,
        private ForwardToMainServer $forwardToMainServer,
        private GameSession $gameSession,
        private User $userModel,
        private Logger $logger,
        private string $proxyBaseUrl,
    ) {}

    public function handle(array $input, string $clientIp, ?string $origin): array
    {
        $apiKey = trim((string) ($input['api_key'] ?? ''));
        $userId = trim((string) ($input['user_id'] ?? ''));
        $balanceVal = (float) ($input['balance'] ?? 0);
        $gameUid = trim((string) ($input['game_uid'] ?? ''));
        $returnUrl = trim((string) ($input['return_url'] ?? $input['return'] ?? ''));
        $callbackUrl = trim((string) ($input['callback_url'] ?? $input['callback'] ?? ''));
        $currencyCode = trim((string) ($input['currency_code'] ?? ''));
        $language = trim((string) ($input['language'] ?? ''));

        $this->logger->log(null, '/api/launch-game', $input, null, $clientIp, null);

        $validation = ($this->validateApiKey)($apiKey);
        if (!$validation['valid']) {
            $err = ['code' => 401, 'msg' => $validation['error']];
            $this->logger->log(null, '/api/launch-game', null, $err, $clientIp, 401);
            return $err;
        }

        $client = $validation['client'];
        if (strtolower(trim((string) ($client['status'] ?? ''))) !== 'active') {
            $err = ['code' => 403, 'msg' => 'Client is inactive'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 403);
            return $err;
        }

        if (!$this->checkWhitelist->__invoke($client, $clientIp, $origin)) {
            $err = ['code' => 403, 'msg' => 'IP or domain not allowed'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 403);
            return $err;
        }

        if (!$this->rateLimiter->check($apiKey)) {
            $err = ['code' => 429, 'msg' => 'Too many requests'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 429);
            return $err;
        }

        if (empty($userId) || empty($gameUid)) {
            $err = ['code' => 400, 'msg' => 'user_id and game_uid are required'];
            $this->logger->log($client['id'], '/api/launch-game', null, $err, $clientIp, 400);
            return $err;
        }

        $clientId = (int) $client['id'];
        $user = $this->userModel->findOrCreate($clientId, $userId, $balanceVal);
        if ($balanceVal > 0) {
            $this->userModel->updateBalance($clientId, $userId, $balanceVal);
        }

        $returnUrl = $returnUrl ?: $client['return_url'] ?? $this->proxyBaseUrl . '/';
        $callbackUrl = $callbackUrl ?: $client['callback_url'] ?? '';

        $userBalance = (float) ($user['balance'] ?? $balanceVal);

        $forwardParams = [
            'user_id' => $userId,
            'balance' => $userBalance > 0 ? $userBalance : $balanceVal,
            'game_uid' => $gameUid,
            'return_url' => $returnUrl,
            'currency_code' => $currencyCode,
            'language' => $language,
        ];

        $response = ($this->forwardToMainServer)($forwardParams);

        $sessionId = $response['decoded']['data']['session_id'] ?? null;
        $this->gameSession->createOrUpdate($clientId, $userId, $gameUid, $sessionId, $callbackUrl, $returnUrl);

        $this->logger->log($clientId, '/api/launch-game', ['payload' => '***'], $response['decoded'] ?? ['raw' => $response['body'] ?? ''], $clientIp, $response['code'] ?? 200);

        if (!$response['success']) {
            return [
                'code' => $response['code'] ?: 500,
                'msg' => $response['error'] ?? ($response['decoded']['msg'] ?? 'Request failed'),
                'data' => $response['decoded']['data'] ?? null,
            ];
        }

        return $response['decoded'] ?? ['code' => 0, 'msg' => 'OK', 'data' => json_decode($response['body'] ?? '{}', true)];
    }
}
