<?php

declare(strict_types=1);

namespace App\Controllers;

use App\Models\CallbackEvent;
use App\Models\CallbackRetry;
use App\Models\Client;
use App\Models\GameSession;
use App\Models\Transaction;
use App\Models\User;
use App\Services\HttpService;
use App\Services\Logger;
use PDO;

class CallbackController
{
    public function __construct(
        private PDO $pdo,
        private GameSession $gameSession,
        private Transaction $transaction,
        private User $userModel,
        private Client $clientModel,
        private CallbackRetry $callbackRetry,
        private CallbackEvent $callbackEvent,
        private HttpService $http,
        private Logger $logger,
    ) {}

    public function handle(string $rawInput, string $clientIp): array
    {
        $data = json_decode($rawInput, true);
        $this->logger->log(null, '/api/callback', $data ?: ['raw' => substr($rawInput, 0, 500)], null, $clientIp, null);

        if (!$data || !is_array($data)) {
            return $this->errorResponse(-1, 'Invalid JSON 无效JSON');
        }

        $required = ['game_uid', 'game_round', 'member_account', 'bet_amount', 'win_amount'];
        foreach ($required as $field) {
            if (!array_key_exists($field, $data)) {
                return $this->errorResponse(-1, "Missing required field: $field");
            }
        }

        $gameUid = (string) $data['game_uid'];
        $gameRound = (string) $data['game_round'];
        $memberAccount = (string) $data['member_account'];
        $betAmount = (float) $data['bet_amount'];
        $winAmount = (float) $data['win_amount'];

        $callbackKey = $gameRound . '|' . $betAmount . '|' . $winAmount . '|' . ($data['timestamp'] ?? '');

        $session = $this->gameSession->findByGameUidAndUser($gameUid, $memberAccount)
            ?? $this->gameSession->findByGameUid($gameUid)
            ?? $this->gameSession->findLatestByUser($memberAccount);
        if (!$session) {
            $this->logger->log(null, '/api/callback', array_merge($data, ['_session_not_found' => true]), null, $clientIp, 404);
            return $this->errorResponse(-1, 'Unknown game session');
        }

        $clientId = (int) $session['client_id'];
        $userId = $session['user_id'];
        $client = $this->clientModel->getById($clientId);
        if (!$client) {
            return $this->errorResponse(-1, 'Client not found');
        }

        $callbackUrl = $session['callback_url'] ?: $client['callback_url'];

        $forwardPayload = array_merge($data, [
            'timestamp' => (int) round(microtime(true) * 1000),
            'callback_key' => $callbackKey,
        ]);

        if ($callbackUrl) {
            $result = $this->http->postJson($callbackUrl, $forwardPayload);
            $this->logger->log($clientId, 'client_callback', $forwardPayload, $result['decoded'] ?? ['raw' => $result['body'] ?? $result['error'] ?? ''], $clientIp, $result['code'] ?? 0);

            if (($result['success'] ?? false) && isset($result['decoded']['credit_amount'])) {
                $response = [
                    'credit_amount' => round((float) $result['decoded']['credit_amount'], 2),
                    'timestamp' => (int) ($result['decoded']['timestamp'] ?? round(microtime(true) * 1000)),
                ];
                if ($response['credit_amount'] >= 0) {
                    $this->userModel->updateBalance($clientId, $userId, $response['credit_amount']);
                }
                $this->logger->log($clientId, '/api/callback', null, $response, $clientIp, 200);
                return $response;
            }
            if (!$result['success'] && $callbackUrl) {
                $this->callbackRetry->add($gameRound, $clientId, $forwardPayload, $callbackUrl);
            }
        }

        $user = $this->userModel->findByClientAndUserId($clientId, $userId);
        $bal = $user ? (float) $user['balance'] : 0;
        $response = ['credit_amount' => round(max(0, $bal), 2), 'timestamp' => (int) round(microtime(true) * 1000)];
        $this->logger->log($clientId, '/api/callback', null, $response, $clientIp, 200);
        return $response;
    }

    private function errorResponse(int $credit, string $message): array
    {
        return ['credit_amount' => $credit, 'error' => $message];
    }
}
